------------------------------------------------------------------------
-- Event:        Delphi Day 2018, Piacenza, June 06 2018               -
--               https://www.delphiday.it/                             -
-- Seminary:     How to write high performance queries in T-SQL        -
-- Demo:         APPLY operator                                        -
-- Author:       Sergio Govoni                                         -
-- Notes:        --                                                    -
------------------------------------------------------------------------

USE [WideWorldImporters];
GO

SET STATISTICS IO ON;
SET STATISTICS TIME ON;
GO


SELECT
  so.OrderID
  ,sol.StockItemID
  ,sol.UnitPrice
  ,WebPrice = Website.DDay18CalculateCustomerPrice(c.BuyingGroupID, sol.StockItemID, sol.UnitPrice, so.OrderDate)
FROM
  Sales.Orders AS so
JOIN
  Sales.OrderLines AS sol ON so.OrderID=sol.OrderID
JOIN
  Sales.Customers AS c ON so.CustomerID=c.CustomerID;
GO


CREATE OR ALTER FUNCTION Website.DDay18CalculateCustomerPriceITVF
(
    @BuyingGroupID INTEGER
    ,@StockItemID INTEGER
    ,@UnitPrice DECIMAL(18, 2)
    ,@PricingDate DATE
)
RETURNS TABLE
AS
RETURN
(
  SELECT
    --MAX(sd.DiscountPercentage) AS DiscountPercentage    
    CAST(ISNULL(@UnitPrice - ROUND(@UnitPrice * MAX(sd.DiscountPercentage) / 100., 2), @UnitPrice) AS DECIMAL(18, 2)) AS DiscountPercentage
  FROM
    Sales.SpecialDeals AS sd
  WHERE
    ((sd.StockItemID = @StockItemID) OR (sd.StockItemID IS NULL))
    AND ((sd.BuyingGroupID = @BuyingGroupID) OR (sd.BuyingGroupID IS NULL))
    AND (@PricingDate BETWEEN sd.StartDate AND sd.EndDate)
);
GO


SELECT
  so.OrderID
  ,sol.StockItemID
  ,sol.UnitPrice
  --,WebPrice = Website.DDay18CalculateCustomerPrice(c.BuyingGroupID, sol.StockItemID, so.OrderDate)
  ,WebPrice.DiscountPercentage
FROM
  Sales.Orders AS so
JOIN
  Sales.OrderLines AS sol ON so.OrderID=sol.OrderID
JOIN
  Sales.Customers AS c ON so.CustomerID=c.CustomerID
CROSS APPLY
  Website.DDay18CalculateCustomerPriceITVF(c.BuyingGroupID, sol.StockItemID, sol.UnitPrice, so.OrderDate) AS WebPrice
GO






---- Altri usi di APPLY
/*
SELECT
  TOP 1
  SupplierID
  ,MaxOrderDate = OrderDate
  ,MaxOrderID = PurchaseOrderID
FROM
  Purchasing.PurchaseOrders
WHERE
  SupplierID = 4
ORDER BY
  OrderDate DESC
  ,PurchaseOrderID DESC;
GO
*/


-- Per ogni cliente, si desidera visualizzare la data dell'ultimo ordine
-- e il numero dell'ultimo ordine nella data

-- Per un singolo cliente, la query  molto semplice,
-- utilizzo l'operatore TOP e la clausola ORDER BY
SELECT
  TOP 1
  CustomerID
  ,MaxOrderDate = OrderDate
  ,MaxOrderID = OrderID
FROM
  Sales.Orders
WHERE
  CustomerID = 4
ORDER BY
  OrderDate DESC -- Data dell'ultimo ordine
  ,OrderID DESC; -- Numero dell'ultimo ordine
GO


SET STATISTICS IO ON;
GO

-- Dovendo estendere l'estrazione a tutti i clienti,
-- utilizzo due subquery
SELECT
  C.CustomerID
  ,MaxOrderDate = (SELECT MAX(O1.OrderDate)
                   FROM Sales.Orders AS O1
                   WHERE O1.CustomerID=C.CustomerID)
  ,MaxOrderID = (SELECT MAX(O2.OrderID)
                 FROM Sales.Orders AS O2
                 WHERE O2.CustomerID=C.CustomerID
                   AND OrderDate = (SELECT MAX(O3.OrderDate)
                                    FROM Sales.Orders AS O3
                                    WHERE O3.CustomerID=C.CustomerID)
				)
FROM
  Sales.Customers AS C
ORDER BY
  C.CustomerID;
GO


-- Se non voglio utilizzare subquery, potrei risolvere
-- il problema con una CTE
WITH Customers AS
(
  SELECT
    CustomerID
  FROM
    Sales.Customers
),
MaxOrderDate AS
(
  SELECT
    O.CustomerID
    ,MaxOrderDate = MAX(O.OrderDate)
  FROM
	  Sales.Orders AS O
  JOIN
    Customers AS C ON C.CustomerID = O.CustomerID
	GROUP BY
	  O.CustomerID
),
MaxOrderDateAndID AS
(
  SELECT
    O.CustomerID
    ,M.MaxOrderDate
    ,MaxOrderID = MAX(OrderID)
  FROM
    Sales.Orders AS O
  JOIN
    MaxOrderDate AS M ON M.CustomerID = O.CustomerID AND M.MaxOrderDate = O.OrderDate
  GROUP BY
    O.CustomerID
    ,M.MaxOrderDate
)
SELECT
  MaxOrderDateAndID.CustomerID
  ,MaxOrderDateAndID.MaxOrderDate
  ,MaxOrderDateAndID.MaxOrderID
FROM
  MaxOrderDateAndID
ORDER BY
  CustomerID;
GO


-- Se le specifiche cambiano e vengono richieste altre colonne?
-- La complessit della query aumenta
WITH Customers AS
(
  SELECT
    CustomerID
  FROM
    Sales.Customers
),
MaxOrderDate AS
(
  SELECT
    O.CustomerID
    ,MaxOrderDate = MAX(O.OrderDate)
  FROM
	  Sales.Orders AS O
  JOIN
    Customers AS C ON C.CustomerID = O.CustomerID
	GROUP BY
	  O.CustomerID
),
MaxOrderDateAndID AS
(
  SELECT
    O.CustomerID
    ,M.MaxOrderDate
    ,MaxOrderID = MAX(OrderID)
  FROM
    Sales.Orders AS O
  JOIN
    MaxOrderDate AS M ON M.CustomerID = O.CustomerID AND M.MaxOrderDate = O.OrderDate
  GROUP BY
    O.CustomerID
    ,M.MaxOrderDate
)
SELECT
  MaxOrderDateAndID.CustomerID
  ,MaxOrderDateAndID.MaxOrderDate
  ,MaxOrderDateAndID.MaxOrderID
  ,PickingCompleted = (
                        SELECT
                          O.PickingCompletedWhen
                        FROM
                          Sales.Orders AS O
                        WHERE
                          O.CustomerID = MaxOrderDateAndID.CustomerID
                          AND O.OrderID = MaxOrderDateAndID.MaxOrderID
                          AND O.OrderDate = MaxOrderDateAndID.MaxOrderDate
                      )
FROM
  MaxOrderDateAndID
ORDER BY
  CustomerID;
GO


-- Le richieste aumentano!
-- Stiamo effettuando molti accessi alle stesse tabelle
WITH Customers AS
(
  SELECT
    CustomerID
  FROM
    Sales.Customers
),
MaxOrderDate AS
(
  SELECT
    O.CustomerID
    ,MaxOrderDate = MAX(O.OrderDate)
  FROM
	  Sales.Orders AS O
  JOIN
    Customers AS C ON C.CustomerID = O.CustomerID
	GROUP BY
	  O.CustomerID
),
MaxOrderDateAndID AS
(
  SELECT
    O.CustomerID
    ,M.MaxOrderDate
    ,MaxOrderID = MAX(OrderID)
  FROM
    Sales.Orders AS O
  JOIN
    MaxOrderDate AS M ON M.CustomerID = O.CustomerID AND M.MaxOrderDate = O.OrderDate
  GROUP BY
    O.CustomerID
    ,M.MaxOrderDate
)
SELECT
  MaxOrderDateAndID.CustomerID
  ,MaxOrderDateAndID.MaxOrderDate
  ,MaxOrderDateAndID.MaxOrderID
  ,PickingCompleted = (
                        SELECT
                          O.PickingCompletedWhen
                        FROM
                          Sales.Orders AS O
                        WHERE
                          O.CustomerID = MaxOrderDateAndID.CustomerID
                          AND O.OrderID = MaxOrderDateAndID.MaxOrderID
                          AND O.OrderDate = MaxOrderDateAndID.MaxOrderDate
                      )
  ,Total = (
             SELECT
               SUM(OL.UnitPrice * OL.PickedQuantity)
             FROM
               Sales.OrderLines AS OL
             WHERE
               OL.OrderID = MaxOrderDateAndID.MaxOrderID
           )
FROM
  MaxOrderDateAndID
ORDER BY
  CustomerID;
GO


-- Per semplificare la query, potrei utilizzare CROSS APPLY..
-- estraggo i clienti e per ognuno effettuo l'estrazione dei
-- dati richiesti. Le colonne aggiuntive non aumenteranno la
-- complessit della query

SELECT
  C.CustomerID
  ,M.MaxOrderDate
  ,M.MaxOrderID
  ,M.PickingCompleted
FROM
  Sales.Customers AS C
CROSS APPLY
  (
    SELECT
      TOP(1)
      CustomerID
      ,MaxOrderDate = OrderDate
      ,MaxOrderID = OrderID
      ,PickingCompleted = PickingCompletedWhen
    FROM
      Sales.Orders
    WHERE
      CustomerID = C.CustomerID
    ORDER BY
      OrderDate DESC
      ,OrderID DESC
  ) AS M
ORDER BY
  C.CustomerID;
GO


-- Diamo un'occhiata al piano di esecuzione..
-- Qual' il problema?

-- SQL Server implementa CROSS APPLY con una JOIN
-- per  costretto ad utilizzare un Nested Loop
-- (di fatto non  riuscito a tradurre il comando in un JOIN)

-- All'interno della query nel CROSS APPLY abbiamo specificato
-- un operatore non relazionale (TOP)



-- Utilizziamo ROW_NUMBER()
SELECT
  C.CustomerID
  ,TMAX.MaxOrderDate
  ,TMAX.MaxOrderID
  ,TMAX.PickingCompleted
FROM
  Sales.Customers AS C
CROSS APPLY
  (
    SELECT *
    FROM (
           SELECT
             RowNumber = ROW_NUMBER() OVER (ORDER BY OrderDate DESC, OrderID DESC)
             ,CustomerID
             ,MaxOrderDate = OrderDate
             ,MaxOrderID = OrderID
             ,PickingCompleted = PickingCompletedWhen
           FROM
             Sales.Orders
           WHERE
             CustomerID = C.CustomerID
         ) AS M
    WHERE
      (M.RowNumber = 1) -- Filtro
  ) AS TMAX
ORDER BY
  C.CustomerID;
GO


-- Il piano di esecuzione non cambia perch il filtro viene
-- applicato all'interno della "funzione" nel CROSS APPLY
-- che viene invocata per ogni cliente




-- Utilizzo ROW_NUMBER al posto di TOP, partizionando per cliente
-- che equivale ad un GROUP BY
SELECT
  C.CustomerID
  ,M.MaxOrderDate
  ,M.MaxOrderID
  ,M.PickingCompleted
FROM
  Sales.Customers AS C
CROSS APPLY
  (
    SELECT
      RowNumber = ROW_NUMBER() OVER (PARTITION BY
                                       CustomerID
                                     ORDER BY
                                       OrderDate DESC
                                       ,OrderID DESC)
      ,CustomerID
      ,MaxOrderDate = OrderDate
      ,MaxOrderID = OrderID
      ,PickingCompleted = PickingCompletedWhen
    FROM
      Sales.Orders
  ) AS M
WHERE
  (RowNumber = 1)
  AND (C.CustomerID = M.CustomerID)
ORDER BY
  C.CustomerID;
GO